package ga.view.aerial;

import ga.core.individual.IIndividual;
import ga.view.appstate.GUIState;
import ga.view.appstate.menu.MenuListener;

import com.jme3.app.Application;
import com.jme3.app.SimpleApplication;
import com.jme3.app.state.AppStateManager;
import com.jme3.audio.AudioRenderer;
import com.jme3.light.AmbientLight;
import com.jme3.material.Material;
import com.jme3.material.RenderState.BlendMode;
import com.jme3.math.ColorRGBA;
import com.jme3.math.Vector3f;
import com.jme3.niftygui.NiftyJmeDisplay;
import com.jme3.scene.Geometry;
import com.jme3.scene.Mesh;
import com.jme3.scene.shape.Box;

import de.lessvoid.nifty.Nifty;
import de.lessvoid.nifty.elements.Element;
import de.lessvoid.nifty.elements.render.TextRenderer;
import de.lessvoid.nifty.screen.Screen;

/**
 * This is an advanced app state which adds gui to the evaluation state.
 * 
 * @param <T>
 *          The generic type of the individuals.
 * 
 * @since 12.08.2012
 * @author Stephan Dreyer
 */
public class AerialUIState<T extends IIndividual<T>> extends GUIState {
  private AerialUIController<T> controller;
  private NiftyJmeDisplay niftyDisplay;
  private Nifty nifty;
  private AudioRenderer audioRenderer;

  private final AerialEvaluationState<T> evaluator;

  private final MenuListener menuListener;

  /**
   * Instantiates a new aerial ui state.
   * 
   * @param evaluator
   *          the evaluator
   * @param menuListener
   *          the menu listener
   */
  public AerialUIState(final AerialEvaluationState<T> evaluator,
      final MenuListener menuListener) {
    this.evaluator = evaluator;
    this.menuListener = menuListener;
  }

  @Override
  public void initialize(final AppStateManager stateManager,
      final Application app) {
    super.initialize(stateManager, app);

    this.audioRenderer = app.getAudioRenderer();

    controller = new AerialUIController<T>(evaluator);
    controller.setMenuListener(menuListener);

    setEnabled(true);

    evaluator.fireNewIndividualRequested();
  }

  /**
   * Sets the individual.
   * 
   * @param individual
   *          the new individual
   */
  public void setIndividual(final T individual) {
    controller.setIndividual(individual);

    final Screen start = nifty.getScreen("start");

    Element element = start.findElementByName("CostsTextField");
    TextRenderer textRenderer = element.getRenderer(TextRenderer.class);
    textRenderer.setText(controller.getCosts());

    element = start.findElementByName("IdTextField");
    textRenderer = element.getRenderer(TextRenderer.class);
    textRenderer.setText(controller.getId());

    element = start.findElementByName("GenerationTextField");
    textRenderer = element.getRenderer(TextRenderer.class);
    textRenderer.setText(controller.getGeneration());
  }

  @Override
  public void setEnabled(final boolean enabled) {
    super.setEnabled(enabled);

    if (enabled) {
      niftyDisplay = new NiftyJmeDisplay(assetManager, inputManager,
          audioRenderer, viewPort);
      nifty = niftyDisplay.getNifty();

      nifty.fromXml("ga/view/ui/aerialUI.xml", "start", controller);

      viewPort.addProcessor(niftyDisplay);
    } else {
      try {
        if (nifty != null) {
          viewPort.removeProcessor(niftyDisplay);
          nifty.exit();
        }
      } catch (final Exception e) {
        // ignore
      }
    }
  }

  @Override
  public void cleanup() {
    try {
      if (nifty != null) {
        viewPort.removeProcessor(niftyDisplay);
        nifty.exit();
      }
    } catch (final Exception e) {
      // ignore
    }
    super.cleanup();
  }

  /**
   * Main method for testing.
   * 
   * @param args
   *          No arguments.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public static void main(final String[] args) {
    new SimpleApplication() {

      @SuppressWarnings({ "rawtypes", "unchecked" })
      @Override
      public void simpleInitApp() {

        flyCam.setDragToRotate(true);

        final Material mat = new Material(assetManager,
            "Common/MatDefs/Light/Lighting.j3md");
        mat.setFloat("Shininess", 5f);
        mat.setBoolean("UseMaterialColors", true);
        mat.setBoolean("UseAlpha", true);
        // teapotMat.setBoolean("UseVertexColor", true);
        mat.setColor("Ambient", ColorRGBA.White.mult(0.8f));
        mat.setColor("Diffuse", new ColorRGBA(1f, 1f, 1f, 1f));
        mat.setColor("Specular", ColorRGBA.White);

        mat.setBoolean("HighQuality", true);

        mat.setReceivesShadows(true);

        mat.getAdditionalRenderState().setBlendMode(BlendMode.Alpha);

        final Mesh m = new Box(Vector3f.ZERO, 2, 2, 2);

        final Geometry geo = new Geometry("Object", m);
        geo.setMaterial(mat);

        rootNode.attachChild(geo);
        rootNode.addLight(new AmbientLight());

        inputManager.setCursorVisible(true);

        stateManager.attach(new AerialUIState(null, null));
      }
    }.start();
  }
}
